<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Database configuration
$host = 'localhost';
$dbname = 'startmine_kyc';
$username = 'your_db_username';
$password = 'your_db_password';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid JSON input']);
    exit;
}

// Validate required fields
$requiredFields = [
    'firstName', 'lastName', 'email', 'phone', 'dateOfBirth', 'nationality',
    'street', 'city', 'state', 'zipCode', 'country', 'occupation', 'sourceOfFunds',
    'idType', 'idNumber', 'idExpiry', 'termsAccepted', 'privacyAccepted'
];

$missingFields = [];
foreach ($requiredFields as $field) {
    if (!isset($input[$field]) || (is_string($input[$field]) && trim($input[$field]) === '')) {
        // Skip SSN validation if not US nationality
        if ($field === 'ssn' && $input['nationality'] !== 'us') {
            continue;
        }
        $missingFields[] = $field;
    }
}

// Check SSN for US users
if ($input['nationality'] === 'us' && (!isset($input['ssn']) || trim($input['ssn']) === '')) {
    $missingFields[] = 'ssn';
}

if (!empty($missingFields)) {
    http_response_code(400);
    echo json_encode([
        'error' => 'Missing required fields',
        'fields' => $missingFields
    ]);
    exit;
}

// Validate email format
if (!filter_var($input['email'], FILTER_VALIDATE_EMAIL)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid email format']);
    exit;
}

// Validate date of birth (must be 18+ years old)
$dob = new DateTime($input['dateOfBirth']);
$today = new DateTime();
$age = $today->diff($dob)->y;

if ($age < 18) {
    http_response_code(400);
    echo json_encode(['error' => 'Must be 18 years or older']);
    exit;
}

// Validate document expiry (must be in the future)
$expiry = new DateTime($input['idExpiry']);
if ($expiry < $today) {
    http_response_code(400);
    echo json_encode(['error' => 'Document must not be expired']);
    exit;
}

// Check if email already exists
$stmt = $pdo->prepare("SELECT id FROM kyc_applications WHERE email = ?");
$stmt->execute([$input['email']]);
if ($stmt->fetch()) {
    http_response_code(409);
    echo json_encode(['error' => 'Email already registered']);
    exit;
}

// Sanitize and prepare data
$data = [
    'first_name' => trim($input['firstName']),
    'last_name' => trim($input['lastName']),
    'email' => strtolower(trim($input['email'])),
    'phone' => trim($input['phone']),
    'date_of_birth' => $input['dateOfBirth'],
    'nationality' => $input['nationality'],
    'ssn' => $input['nationality'] === 'us' ? trim($input['ssn']) : null,
    'street' => trim($input['street']),
    'city' => trim($input['city']),
    'state' => trim($input['state']),
    'zip_code' => trim($input['zipCode']),
    'country' => $input['country'],
    'occupation' => trim($input['occupation']),
    'source_of_funds' => $input['sourceOfFunds'],
    'expected_transaction_volume' => $input['expectedTransactionVolume'] ?? null,
    'id_type' => $input['idType'],
    'id_number' => trim($input['idNumber']),
    'id_expiry' => $input['idExpiry'],
    'terms_accepted' => $input['termsAccepted'] ? 1 : 0,
    'privacy_accepted' => $input['privacyAccepted'] ? 1 : 0,
    'marketing_accepted' => $input['marketingAccepted'] ? 1 : 0,
    'status' => 'pending',
    'created_at' => date('Y-m-d H:i:s'),
    'updated_at' => date('Y-m-d H:i:s')
];

try {
    // Insert KYC application
    $sql = "INSERT INTO kyc_applications (
        first_name, last_name, email, phone, date_of_birth, nationality, ssn,
        street, city, state, zip_code, country, occupation, source_of_funds,
        expected_transaction_volume, id_type, id_number, id_expiry,
        terms_accepted, privacy_accepted, marketing_accepted,
        status, created_at, updated_at
    ) VALUES (
        :first_name, :last_name, :email, :phone, :date_of_birth, :nationality, :ssn,
        :street, :city, :state, :zip_code, :country, :occupation, :source_of_funds,
        :expected_transaction_volume, :id_type, :id_number, :id_expiry,
        :terms_accepted, :privacy_accepted, :marketing_accepted,
        :status, :created_at, :updated_at
    )";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($data);
    
    $applicationId = $pdo->lastInsertId();
    
    // Log the submission
    error_log("KYC Application submitted: ID {$applicationId}, Email: {$data['email']}");
    
    // Send success response
    http_response_code(201);
    echo json_encode([
        'success' => true,
        'message' => 'KYC application submitted successfully',
        'application_id' => $applicationId,
        'status' => 'pending'
    ]);
    
} catch (PDOException $e) {
    error_log("KYC Database Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Failed to process application']);
}
?>
